using System;
using System.Collections;
using System.Data;

using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stanislav Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/28/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// Simple definition of national facility encapsulating data 
	/// necessary to present facility selection to user.
	/// </summary>
	
	#endregion	
	
	public class FacilityDefinition : IRecognizableByID
	{
		/// <summary>
		/// Standard comparer used to compare national facilities by name.
		/// </summary>
		private class FacilityNameComparer : BusinessObjectPropertyComparerBase
		{
			protected override int CompareProperties( object x, object y )
			{
				return Comparer.DefaultInvariant.Compare( ((FacilityDefinition)x).FacilityName, ((FacilityDefinition)y).FacilityName );
			}
		}

		// Used as a separator between code and name in combined facility name.
		private const string FdaRegNumNameDisplayDivider = " - ";

		private readonly Guid _facilityGuid;
		private readonly string _facilityName;
		private readonly string _fdaRegNumber;

		/// <summary>
		/// Creates a new instance of the class from a supplied <see cref="DataRow"/>
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to fetch object data from.</param>
		protected FacilityDefinition( DataRow dtRow )
		{
			Utility.RequireNonNullColumns( dtRow, VbecsTables.Facility.FacilityGuid, VbecsTables.Facility.FacilityName, VbecsTables.Facility.FdaRegistrationNumber );

			_facilityGuid = (Guid)(dtRow[ VbecsTables.Facility.FacilityGuid ]);
			_facilityName = Convert.ToString( dtRow[ VbecsTables.Facility.FacilityName ] ).Trim();
			_fdaRegNumber = Convert.ToString( dtRow[ VbecsTables.Facility.FdaRegistrationNumber ] ).Trim();
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5626"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-empty GUID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5627"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// National facility GUID.
		/// </summary>
		public Guid FacilityGuid
		{
			get
			{
				return _facilityGuid;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5624"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null combined name.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5625"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// National facility combined name (concatenated facility FDA registration number and name).
		/// </summary>
		public string CombinedName
		{
			get
			{
				return String.Concat( _fdaRegNumber, FdaRegNumNameDisplayDivider, _facilityName );
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5622"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null name.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5623"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// National facility GUID.
		/// </summary>
		public string FacilityName
		{
			get
			{
				return _facilityName;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6305"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null FDA registration number.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6306"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///		The FDA Registration Number for the facility.
		/// </summary>
		public string FdaRegistrationNumber
		{
			get
			{
				return _fdaRegNumber;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6307"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null non-empty list of facilities.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6308"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves all available facilities sorted by name.
		/// </summary>
		/// <returns>List of all available facilities sorted by name.</returns>
		public static BusinessObjectSortedHashList GetAllSortedByName()
		{
			DataTable _dt = BOL.LocalSupplier.GetFacilityNationalLookupList();
			BusinessObjectSortedHashList _list = new BusinessObjectSortedHashList( new FacilityNameComparer(), _dt.Rows.Count );

			foreach( DataRow _dr in _dt.Rows )
				_list.Add( new FacilityDefinition( _dr ) );

			return _list;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6309"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6310"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets/constructs facility object by GUID. 
		/// </summary>
		/// <param name="facilityGuid">Facility GUID to look up for.</param>
		/// <returns>Facility definition object that has appropriate GUID. <see cref="Common.DataAccessLayerException"/> if it's not found.</returns>
		public static FacilityDefinition GetByGuid( Guid facilityGuid )
		{
			return new FacilityDefinition( DAL.LocalSupplier.GetFacilityDefinitionByGuid( facilityGuid ) );
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6311"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid facility GUID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6312"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns unique object ID (facility GUID).
		/// </summary>
		/// <returns>Unique object ID (facility GUID).</returns>
		public object GetObjectID()
		{
			return FacilityGuid;
		}
	}
}
